/**************************************************************************/
/* FILE NAME: FlexCAN_setup.c               COPYRIGHT (c) Freescale 2006  */
/*                                                All Rights Reserved     */
/* DESCRIPTION:                                                           */
/* This file contains the necessary setup code for configuring the FlexCAN*/
/* modules for transmit and receive to external node					  */
/*========================================================================*/
/* REV      AUTHOR       DATE       DESCRIPTION OF CHANGE                 */
/* ---   -----------   ----------   ---------------------                 */
/* 0.2   D.Paterson     01/Feb/07    Integrated to multi-mod test         */
/**************************************************************************/
#include "mpc5561.h"

// Message buffer definition CAN A
#define send_buff_A 0
#define rec_buff_A 1

// Message buffer definition CAN C
#define send_buff_C 2
#define rec_buff_C 3

void setup_FlexCAN(void);
void init_CANA(void);
void init_CANC(void);

unsigned int FlexCANA_test(void);
unsigned int FlexCANC_test(void);

extern int highest_freq;
extern int highest_freq1;

unsigned int passed;
unsigned int flag_check;
unsigned int flag;

int test_fail[4] = {0,0,0,0};
int can_complete;

/*************************************************************************
*	Function Name: 	FlexCAN_test										 *
*	Description: 	This calls CAN A and C to transmit result of FFT	 *
*	Called By :		edma_ch3_ISR										 *
**************************************************************************/ 
void FlexCAN_test (void)
{
    can_complete = 0;

	FlexCANA_test();		//Initialize buffers and transmit data on FlexCAN A
	FlexCANC_test();		//Initialize buffers and transmit data on FlexCAN C
}


/*************************************************************************
*	Function Name: 	setup_FlexCAN										 *
*	Description: 	Configures CAN A and CAN C							 *
*	Called By :		setup_functions										 *
**************************************************************************/ 
void setup_FlexCAN(void)
{
	//Enable FlexCAN transmit and receive pins
    SIU.PCR[83].R = 0x0633;      //CNTXA
    SIU.PCR[84].R = 0x0533;      //CNRXA
    
    SIU.PCR[87].R = 0x0633;      //CNTXC
    SIU.PCR[88].R = 0x0533;      //CNRXC

    init_CANA();				 //Initialize FlexCAN A module
    init_CANC();				 //Initialize FlexCAN C module
}
    
/*************************************************************************
*	Function Name: 	init_CANA										 	 *
*	Description: 	This function sets up FlexCANA module				 *
*	Called By :		setup_FlexCAN										 *
**************************************************************************/ 
void init_CANA(void)
{
	int j;
	
	/* Turn on CAN_A*/
	CAN_A.MCR.R = 0x1080003F;

	/* Initialize all operating modes for CAN_A*/
	CAN_A.CR.R = 0x0f492002;
	
	/* Initialise all buffers to inactive */
    for (j=0; j<64; j++)
    {
           if((j != send_buff_A) && (j != rec_buff_A))
             CAN_A.BUF[j].CS.B.CODE = 0x0;
        }   
    
    /* Enable Rx buffer interrupts */
    CAN_A.IMRL.R = 0x2;		// MB1
    
    /* Set transmit length of CAN_A[0]  */
    CAN_A.BUF[send_buff_A].CS.B.LENGTH = 0x8;
    CAN_A.BUF[send_buff_A].CS.B.LENGTH = 0x8;

    /* Set MB IDE, RTR, and IDs for CAN_A */
    CAN_A.BUF[send_buff_A].CS.B.IDE = 0;
    CAN_A.BUF[rec_buff_A].CS.B.IDE = 0;
    CAN_A.BUF[send_buff_A].CS.B.RTR = 0;
    CAN_A.BUF[rec_buff_A].CS.B.RTR = 0;
        
    CAN_A.BUF[send_buff_A].ID.R = 0x10000000;
    CAN_A.BUF[rec_buff_A].ID.R = 0x08000000;	
}    
    
/*************************************************************************
*	Function Name: 	init_CANC										 	 *
*	Description: 	This function sets up FlexCANC module				 *
*	Called By :		setup_FlexCAN										 *
**************************************************************************/ 
void init_CANC(void)
{
	int j;
	
	/* Turn on CAN_C*/
	CAN_C.MCR.R = 0x1080003F;

	/* Initialize all operating modes for CAN_C*/
	CAN_C.CR.R = 0x0f492002;
	
	/* Initialise all buffers to inactive */
    for (j=0; j<64; j++)
    {
       if((j != send_buff_C) && (j != rec_buff_C))
       		CAN_C.BUF[j].CS.B.CODE = 0x0;
    }   
    
    /* Enable Rx buffer interrupts */
    CAN_C.IMRL.R = 0x8;		// MB3
    
    /* Set transmit length of CAN_C[0]  */
    CAN_C.BUF[send_buff_C].CS.B.LENGTH = 0x8;
    CAN_C.BUF[send_buff_C].CS.B.LENGTH = 0x8;

    /* Set MB IDE, RTR, and IDs for CAN_C */
    CAN_C.BUF[send_buff_C].CS.B.IDE = 0;
    CAN_C.BUF[rec_buff_C].CS.B.IDE = 0;
    CAN_C.BUF[send_buff_C].CS.B.RTR = 0;
    CAN_C.BUF[rec_buff_C].CS.B.RTR = 0;
        
    CAN_C.BUF[send_buff_C].ID.R = 0x11000000;
    CAN_C.BUF[rec_buff_C].ID.R = 0x09000000;	
}
        

/*************************************************************************
*	Function Name: 	FlexCANA_test										 *
*	Description: 	This function transmits to FlexCAN A on node 2		 *
*	Called By :		FlexCAN_test										 *
**************************************************************************/ 
unsigned int FlexCANA_test(void)
{	
    int fail = 0;
	int j=0;

    /* Initialize all CAN_A message buffers as inactive */
    CAN_A.BUF[send_buff_A].CS.B.CODE = 0x0;
    CAN_A.BUF[rec_buff_A].CS.B.CODE = 0x0;
    
    /* Copy tx data to CAN_A[0]*/
    CAN_A.BUF[send_buff_A].DATA.W[0] = 0x01020304;
    CAN_A.BUF[send_buff_A].DATA.W[1] = 0x05060708;

    /*Clear all non-transmit MB space in CAN_A*/
	CAN_A.BUF[rec_buff_A].DATA.W[0] = 0x00000000;
	CAN_A.BUF[rec_buff_A].DATA.W[1] = 0x00000000;

    /* Initialize CAN_A[1] to Receive 	 */
    CAN_A.BUF[rec_buff_A].CS.B.CODE = 0x4;

    /* Mask interrupts */
    CAN_A.RXGMASK.R = 0x1FFFFFFF;

    /* Negate halt bit in MCR to synch with FlexCAN bus */
    CAN_A.MCR.B.HALT = 0;

    /* Transmit data in CAN_A[0] */
    CAN_A.BUF[send_buff_A].CS.B.CODE = 0xC;

}	

/*************************************************************************
*	Function Name: 	cana_rec										 	 *
*	Description: 	This function checks that the data received is valid *
*	Called By :		cana_buf1_ISR										 *
**************************************************************************/ 	
void cana_rec(void)
{
	CAN_A.IFRL.R = 0xFFFFFFFF;
	CAN_A.IFRH.R = 0xFFFFFFFF;

	//Check that reveived data = DSPI transmitted data
    if (CAN_A.BUF[rec_buff_A].DATA.W[0] != highest_freq)	
    {
		test_fail[0] = 1;
    }
    
    if (CAN_A.BUF[send_buff_A].DATA.W[1] != CAN_A.BUF[rec_buff_A].DATA.W[1])
    {
		test_fail[1] = 1;
    }

    // Clear screen and output results 
	printf("\033[2J");
	
	// Display results over eSCI
    if (test_fail[0]==1)
    {
	   	printf("Test has failed -FlexCAN_A \n");
	   	while(1);
	}
	else
	{
	   	printf("Frequency on AN1 : %d \n", highest_freq);
    }
	
}

/*************************************************************************
*	Function Name: 	FlexCANC_test										 *
*	Description: 	This function transmits to FlexCAN C on node 2		 *
*	Called By :		FlexCAN_test										 *
**************************************************************************/ 
unsigned int FlexCANC_test(void)
{	
    int fail = 0;
	int j=0;
	
    /* Initialize all CAN_A message buffers as inactive */
    CAN_C.BUF[send_buff_C].CS.B.CODE = 0x0;
    CAN_C.BUF[rec_buff_C].CS.B.CODE = 0x0;
    
    /* Copy tx data to CAN_A[0]*/
    CAN_C.BUF[send_buff_C].DATA.W[0] = 0x01020304;
    CAN_C.BUF[send_buff_C].DATA.W[1] = 0x05060708;

    /*Clear all non-transmit MB space in CAN_A*/
	CAN_C.BUF[rec_buff_C].DATA.W[0] = 0x00000000;
	CAN_C.BUF[rec_buff_C].DATA.W[1] = 0x00000000;

    /* Initialize CAN_A[1] to Receive 	 */
    CAN_C.BUF[rec_buff_C].CS.B.CODE = 0x4;

    /* Mask interrupts */
    CAN_C.RXGMASK.R = 0x1FFFFFFF;

    /* Negate halt bit in MCR to synch with FlexCAN bus */
    CAN_C.MCR.B.HALT = 0;

    /* Transmit data in CAN_A[0] */
    CAN_C.BUF[send_buff_C].CS.B.CODE = 0xC;

}

/*************************************************************************
*	Function Name: 	canc_rec										 	 *
*	Description: 	This function checks that the data received is valid *
*	Called By :		canc_buf3_ISR										 *
**************************************************************************/ 
void canc_rec(void)
{
	CAN_C.IFRL.R = 0xFFFFFFFF;
	CAN_C.IFRH.R = 0xFFFFFFFF;

    //Check that reveived data = DSPI transmitted data
    
	if (highest_freq1 != CAN_C.BUF[rec_buff_C].DATA.W[0])	
    {
		test_fail[2] = 1;
    }

    if (CAN_C.BUF[send_buff_C].DATA.W[1] != CAN_C.BUF[rec_buff_C].DATA.W[1])
    {
		test_fail[3] = 1;
    }

    // Display results over eSCI
    if (test_fail[2]==1)
    {
	   	printf("Test has failed -FlexCAN_C \n");
	   	while(1);
	}
	else
	{
	   	printf("Frequency on AN2 : %d \n", highest_freq1);
    }
    
    // Set CAN complete flag
    can_complete = 1;

}